-- liman_defaults.lua - Extract default values from liman.h
-- This module parses liman.h to get LIMPAR_* constants, ensuring
-- Lua scripts use the same defaults as the C code (single source of truth).
--
-- Usage:
--   local defaults = require("liman_defaults")
--   local keysize = defaults.LIMPAR_DEFAULT_KEYSIZE
--   local hashsize = defaults.LIMPAR_DEFAULT_HASHSIZE
--
-- (c) Baroks Inc.

local M = {}

-- Default values (fallbacks if parsing fails)
local FALLBACK_DEFAULTS = {
    LIMPAR_DEFAULT_KEYSIZE = 3072,
    LIMPAR_DEFAULT_HASHSIZE = 256,
    LIMPAR_MAX_PATH = 2048,
    LIMPAR_MAX_CHARS_PER_LINE = 256,
    LIMPAR_MAX_LIC_STRING_LEN = 4096,
    LIMPAR_NUM_FEATURE_SLOTS = 4,
}

-- Find liman.h relative to this script or common locations
local function find_liman_h()
    local paths = {
        -- Relative to apps/lua/
        "../../include/liman.h",
        -- Relative to lib/<platform>/
        "../include/liman.h",
        "../../include/liman.h",
        -- Absolute fallback
        "/home/mka/dev/c/liman/trunk/include/liman.h",
    }

    for _, path in ipairs(paths) do
        local f = io.open(path, "r")
        if f then
            f:close()
            return path
        end
    end
    return nil
end

-- Parse a #define line and extract name and numeric value
local function parse_define(line)
    -- Match: #define NAME VALUE
    -- Handle both decimal and hex values
    local name, value = line:match("^%s*#define%s+([%w_]+)%s+(%S+)")
    if name and value then
        -- Try to convert to number
        local num = tonumber(value)
        if not num then
            -- Try hex format
            num = tonumber(value:match("0[xX](%x+)"), 16)
        end
        return name, num
    end
    return nil, nil
end

-- Parse liman.h and extract LIMPAR_* constants
function M.parse_liman_h(filepath)
    filepath = filepath or find_liman_h()
    if not filepath then
        io.stderr:write("Warning: liman.h not found, using fallback defaults\n")
        return FALLBACK_DEFAULTS
    end

    local f = io.open(filepath, "r")
    if not f then
        io.stderr:write("Warning: Cannot open " .. filepath .. ", using fallback defaults\n")
        return FALLBACK_DEFAULTS
    end

    local constants = {}
    for line in f:lines() do
        local name, value = parse_define(line)
        if name and value and name:match("^LIMPAR_") then
            constants[name] = value
        end
    end
    f:close()

    -- Merge with fallbacks for any missing values
    for k, v in pairs(FALLBACK_DEFAULTS) do
        if not constants[k] then
            constants[k] = v
        end
    end

    return constants
end

-- Load constants on module load
local constants = M.parse_liman_h()

-- Export individual constants for easy access
M.LIMPAR_DEFAULT_KEYSIZE = constants.LIMPAR_DEFAULT_KEYSIZE
M.LIMPAR_DEFAULT_HASHSIZE = constants.LIMPAR_DEFAULT_HASHSIZE
M.LIMPAR_MAX_PATH = constants.LIMPAR_MAX_PATH
M.LIMPAR_MAX_CHARS_PER_LINE = constants.LIMPAR_MAX_CHARS_PER_LINE
M.LIMPAR_MAX_LIC_STRING_LEN = constants.LIMPAR_MAX_LIC_STRING_LEN
M.LIMPAR_NUM_FEATURE_SLOTS = constants.LIMPAR_NUM_FEATURE_SLOTS

-- Convenience aliases
M.DEFAULT_KEYSIZE = M.LIMPAR_DEFAULT_KEYSIZE
M.DEFAULT_HASHSIZE = M.LIMPAR_DEFAULT_HASHSIZE

-- Get all constants as a table
function M.get_all()
    return constants
end

-- Print all constants (for debugging)
function M.print_all()
    print("LIMAN Constants from liman.h:")
    print("-----------------------------")
    for k, v in pairs(constants) do
        print(string.format("  %-30s = %s", k, tostring(v)))
    end
end

return M
