---
-- limgen.lua
-- This is a utility application to generate and verify
-- license keys with specified features.
--
-- @remark Derived from apps/c/limgen.c
package.cpath = os.getenv("LIMAN_ROOT") .. "/bin/win32x86/?.dll"

local liman = require 'lualiman'
local defaults = require 'liman_defaults'

---
-- C-sprintf
sprintf = function(fmt, ...)   
    local s=string.format(fmt, ...)
    return s 
end
---
-- C-printf
printf = function(fmt, ...) 
    io.stdout:flush() 
    io.write(string.format(fmt, ...)) 
    io.stdout:flush() 
end 

string.splitz = function(str, delim)
  local delim = delim or "%s"
  local pattern = pattern or "[^" .. delim .. "]+"
  if pattern:len() == 0 then pattern = "[^" .. delim .. "]+" end
  local parts = {__index = table.insert}
  setmetatable(parts, parts)
  str:gsub(pattern, parts)
  setmetatable(parts, nil)
  parts.__index = nil
  return parts
end


---
-- Check error 
function checkerr(pEnv,nErr,isExit)
	local isExit = isExit or 0
	if nErr>0 then
		printf("Error %d: %s\n",nErr,liman.lim_errmsg(pEnv,nErr))
    if isExit>0 then
      printf("Exiting (%d)..\n",isExit);
      nErr = liman.lim_delete_env(pEnv)
      os.exit(1)
    end		
  end
end

---
-- Main
local nErr,major,minor,build,rev,sbuf = liman.lim_get_verkey(nil)
printf("\nLIMAN SDK %d.%d.%d %s\n",major,minor,rev,sbuf)

-- create product-license-base
local szProductId="MYLUAPP"
local nKeySize=defaults.DEFAULT_KEYSIZE
local nSeed=1031


nErr = liman.lim_gen_keypair(nKeySize,nSeed,szProductId);
checkerr(pEnv,nErr,0)

local szPublicKeyFile="MYLUAPP/MYLUAPP_public.key"
local szPrivateKeyFile="MYLUAPP/MYLUAPP_private.key"


-- new license env
local nHashSize=defaults.DEFAULT_HASHSIZE
local pEnv,nErr = liman.lim_create_env(nKeySize,nHashSize)
checkerr(pEnv,nErr,0)


--prep encrypted messages from MYLUAPP users (to be sent to you)
nErr = liman.lim_read_public_key(pEnv,szPublicKeyFile); 
checkerr(pEnv,nErr,0)
nErr = liman.lim_read_private_key(pEnv,szPrivateKeyFile);
checkerr(pEnv,nErr,0)

if arg[1] and string.find(arg[1],"--licinfo") then
  -- read existing license from file
  local tokens = arg[1]:splitz("=")
  szLicenseKeyFile = tokens[2]
   
  -- read back the license key and verify
  pLic, nErr = liman.lim_create_lic_fromfile(pEnv,szLicenseKeyFile);
  checkerr(pEnv,nErr,0)
  -- print license out
  nErr, strbuf = liman.lim_print_lic(pLic,nil);
  checkerr(pEnv,nErr,0)
  printf('\n%s\n',strbuf)
  
   
  nErr = liman.lim_write_lickey_carr(pLic,nil);
  checkerr(pEnv,nErr,0)
   
  printf("Done processing license info\n")
  return
end

-- create a new license
pLic,nErr = liman.lim_create_lic(pEnv);

local szEmail,szlicfile, szSerial
local userid, hostid

-- optional data (e.g. customer-id)
szEmail = "me@gmail.com"
printf("\n")
nErr, szCustomerId = liman.lim_get_hash12(szEmail,string.len(szEmail),nil);
checkerr(pEnv,nErr,0)
printf("CustomerId  : %s\n",szCustomerId) 

local userid = os.getenv('USERNAME') or os.getenv('USER')
printf("userid    : %s\n",userid)

--serial
nErr, szSerial = liman.lim_gen_rkey(pEnv,0,liman.LIMPAR_MAX_LEN_SERIAL,32,nil);
checkerr(pEnv,nErr,0)
printf("szSerial  : %s\n",szSerial)

--hostid
nErr,hostid = liman.lim_get_hostid(nil,0);
checkerr(pEnv,nErr,0)
printf("hostid    : %s\n",hostid)

--uuid
printf("\n")
local uuid
nErr,uuid = liman.lim_gen_uuid(pEnv,0,nil) -- seed is time()

local mPlatformId,nMajorVersion,nMinorVersion = liman.LIMPLAT_WIN32X86, 3, 1
local mSizeId,mTypeId,nUsers,nExpiration = liman.LIMSIZE_DEMO, liman.LIMTYPE_EDUCATIONAL, 1, 20200623
local nHostIdMask = 0

local i1 = liman.LIMOPT_FEATURE_0 + liman.LIMOPT_FEATURE_1 + liman.LIMOPT_FEATURE_7 -- 2^0 + 2^1 + 2^7  = 1 + 2 + 128 = 131
local i2 = liman.LIMOPT_FEATURE_1 + liman.LIMOPT_FEATURE_2 + liman.LIMOPT_FEATURE_0 -- 2^1 + 2^2 + 2^0  = 2 + 4 + 1   = 7
local i3 = 255
local i4 = 255
local features = string.char(i1, i2, i3, i4)

 -- generate a license key
 nErr = liman.lim_load_lic(pLic,szProductId,mPlatformId,nMajorVersion,nMinorVersion,
   mSizeId,mTypeId,nUsers,nExpiration,features,szSerial,uuid,hostid,userid,nHostIdMask);
 checkerr(pEnv,nErr,1)

 -- write/export license key to stdout
 nErr = liman.lim_write_lickey(pLic,nil);
 checkerr(pEnv,nErr,0)
 --nErr = liman.lim_write_lickey_carr(pLic);
 --checkerr(pEnv,nErr,0)
 szlicfile = sprintf("%s/usr_%s.lic",szProductId,liman.lim_lic_id(pLic));
 nErr = liman.lim_write_lickey(pLic,szlicfile); 
 printf("\nWritten license to '%s'\n",szlicfile)
 
 printf("Destroying license '%s'\n",liman.lim_lic_id(pLic));
 -- destroy instance
 --nErr = liman.lim_free_lic(pLic);
 checkerr(pEnv,nErr,0)
 pLic=nil
 
 -- read back the license key and verify
 pLic, nErr = liman.lim_create_lic_fromfile(pEnv,szlicfile);
 checkerr(pEnv,nErr,0)
 printf("Created successfully a new instance from '%s'..\n",szlicfile);

 local strbuf
 printf("Displaying license properties and features..\n");
 nErr,strbuf = liman.lim_print_lic(pLic,nil); 
 printf('\n%s\n',strbuf)
 
nErr = liman.lim_delete_env(pEnv)
checkerr(0,nErr,0)

