/** \example testapp.java

 * \brief
 * This is a test application illustrating how to
 * integrate LIMAN SDK to your software to protect
 * it with license keys you generate with 'limgen'
 * or a derivative of it.
 *
 * It is also illustrated how license-features can
 * be used to let you impose license-based restrictions
 * on certain capabilities of your software.
 */

import com.baroks.liman. * ;

/**
   @brief Runnable class with dummy sort routines and dummy features
          using LIMAN SDK for license protection.
*/
public class testapp implements Runnable

{

    private static String szLicFile; //!< license file name

    private static String szProduct; //!< User-specified product-code string

    private static String szPublicKeyFile; //!< Full-path to public key file

    private static String szPrivateKeyFile; //!< Full-path to private key file (for license generation or verification)

	//!< 0'th slot specifies which sorting methods are available under given license key
	private static int slotIdSorting = 0;

	//!< 1'st slot specifies which app functions are available under given license key
	private static int slotIdAppFunc = 1;

    private static String OS = System.getProperty("os.name").toLowerCase();

    static
    {


        /// @brief Load LIMAN SDK dynamic libarary
        try {
			if (OS.contains("win")) {
            	System.loadLibrary("libliman_sdk");
			} else {
				System.loadLibrary("liman_sdk"); //prefix <lib> is implied on non-windows
			}
        }
        catch (Exception e)
        {
            String limanRoot = System.getenv("LIMAN_ROOT");

            if (limanRoot == null) {
                System.out.println("Error: LIMAN_ROOT environment variable is not defined.. ");
            } else {
				try {
					System.load(limanRoot + "/lib/linux64x86/libliman_sdk.so");
				}

				catch (Exception e2) {
					System.load(limanRoot + "/lib/osx64x86/libliman_sdk.jnilib");
				}
		    }
        }

    }

    //! [Sort Functions]
	/** \brief Quick Sort
	@param pLic A LIMAN license instance
	@param array Integer array to sort
	*/
    public int Quick_Sort(SWIGTYPE_p_LIMLIC pLic, int [] array)

    {

        if (0 == liman.lim_has_feature(pLic, slotIdSorting,  liman.LIMOPT_FEATURE_0)) {

            return liman.LIMERR_NOT_SUPPORTED;

            // feature is not available

        }

        /* INSERT the code to perform the sort */

        return 0;

    }

	/** \brief Merge Sort
	@param pLic A LIMAN license instance
	@param array Integer array to sort
	*/
    public int Merge_Sort(SWIGTYPE_p_LIMLIC pLic, int [] array)

    {

        if (0 == liman.lim_has_feature(pLic, slotIdSorting,  liman.LIMOPT_FEATURE_1)) {

            return liman.LIMERR_NOT_SUPPORTED;

            // feature is not available

        }

        /* INSERT the code to perform the sort */

        return 0;

    }

	/** \brief Heap Sort
	@param pLic A LIMAN license instance
	@param array Integer array to sort
	*/
    public int Heap_Sort(SWIGTYPE_p_LIMLIC pLic, int [] array)

    {

        if (0 == liman.lim_has_feature(pLic, slotIdSorting,  liman.LIMOPT_FEATURE_2)) {

            return liman.LIMERR_NOT_SUPPORTED;

            // feature is not available

        }

        /* INSERT the code to perform the sort */

        return 0;

    }

	/** \brief Insertion Sort
	@param pLic A LIMAN license instance
	@param array Integer array to sort
	*/
    public int Insertion_Sort(SWIGTYPE_p_LIMLIC pLic, int [] array)

    {

        if (0 == liman.lim_has_feature(pLic, slotIdSorting,  liman.LIMOPT_FEATURE_3)) {

            return liman.LIMERR_NOT_SUPPORTED;

            // feature is not available

        }

        /* INSERT the code to perform the sort */

        return 0;

    }

	/** \brief Intro Sort
	@param pLic A LIMAN license instance
	@param array Integer array to sort
	*/
    public int Intro_Sort(SWIGTYPE_p_LIMLIC pLic, int [] array)

    {

        if (0 == liman.lim_has_feature(pLic, slotIdSorting,  liman.LIMOPT_FEATURE_4)) {

            return liman.LIMERR_NOT_SUPPORTED;

            // feature is not available

        }

        /* INSERT the code to perform the sort */

        return 0;

    }

	/** \brief Block Sort
	@param pLic A LIMAN license instance
	@param array Integer array to sort
	*/
    public int Block_Sort(SWIGTYPE_p_LIMLIC pLic, int [] array)

    {

        if (0 == liman.lim_has_feature(pLic, slotIdSorting,  liman.LIMOPT_FEATURE_5)) {

            return liman.LIMERR_NOT_SUPPORTED;

            // feature is not available

        }

        /* INSERT the code to perform the sort */

        return 0;

    }

	/** \brief Shell Sort
	@param pLic A LIMAN license instance
	@param array Integer array to sort
	*/
    public int Shell_Sort(SWIGTYPE_p_LIMLIC pLic, int [] array)

    {

        if (0 == liman.lim_has_feature(pLic, slotIdSorting,  liman.LIMOPT_FEATURE_6)) {

            return liman.LIMERR_NOT_SUPPORTED;

            // feature is not available

        }

        /* INSERT the code to perform the sort */

        return 0;

    }

	/** \brief Bubble Sort
	@param pLic A LIMAN license instance
	@param array Integer array to sort
	*/
    public int Bubble_Sort(SWIGTYPE_p_LIMLIC pLic, int [] array)

    {

        if (0 == liman.lim_has_feature(pLic, slotIdSorting,  liman.LIMOPT_FEATURE_7)) {

            return liman.LIMERR_NOT_SUPPORTED;

            // feature is not available

        }

        /* INSERT the code to perform the sort */

        return 0;

    }

    //! [App Functions]

    /**
     *  \brief some application function
     *         requires the license to have feature[1] to have 1st and 2nd bits turned on (bitmasks 2^0 and 2^1)
	 * @brief pLic A LIMAN license instance
     */
    public int app_function0(SWIGTYPE_p_LIMLIC pLic)
    {
        // this function requires feature[1] 1st and 2nd bits turned on
        if (0 == liman.lim_has_feature(pLic, slotIdAppFunc, liman.LIMOPT_FEATURE_0) || 0 == liman.lim_has_feature(pLic, slotIdAppFunc, liman.LIMOPT_FEATURE_1))
        {
            //license does not support
            System.out.println("app_function0 is not allowed with existing license");
            return liman.LIMERR_NOT_SUPPORTED;
        }

        //proceed with feature
        System.out.println("app_function0 is allowed with existing license");

        /* INSERT FUNCTION-2 CODE */

        return 0;

    }

    /**
     *  \brief another application function
     *         requires the license to have feature[1] to have 1st bit turned on (bitmasks 2^0 )
	 * @brief pLic A LIMAN license instance
     */
    public int app_function1(SWIGTYPE_p_LIMLIC pLic)
    {
        // this function requires feature[1] 1st bit turned on
        if (0 == liman.lim_has_feature(pLic, slotIdAppFunc, liman.LIMOPT_FEATURE_0))
        {
            //license does not support
            System.out.println("app_function1 is not allowed with existing license");
            return liman.LIMERR_NOT_SUPPORTED;
        }

        //proceed with feature
        System.out.println("app_function1 is allowed with existing license");

        /* INSERT FUNCTION-1 CODE */

        return 0;

    }

    /**
     *  \brief another application function
     *         requires the license to have feature[1] to have 1st and 5th bits turned on (bitmasks 2^0 and 2^4)
	 * @brief pLic A LIMAN license instance
     */
    public int app_function2(SWIGTYPE_p_LIMLIC pLic)
    {
        // this function requires feature[1] 1st and 5th bits turned on
        if (0 == liman.lim_has_feature(pLic, slotIdAppFunc, liman.LIMOPT_FEATURE_0) || 0 == liman.lim_has_feature(pLic, slotIdAppFunc, liman.LIMOPT_FEATURE_4))
        {
            //license does not support
            System.out.println("app_function2 is not allowed with existing license");
            return liman.LIMERR_NOT_SUPPORTED;
        }

        //proceed with feature

        System.out.println("app_function2 is allowed with existing license");

        /* INSERT FUNCTION-2 CODE */

        return 0;

    }

    public testapp() { }

	/**
	\brief Main testapp
	\param args command-line arguments
	*/
    public static void main(final String[] args)
    {
        if (args.length < 2) {
            System.out.println("\nUsage: testapp [product_id] [license_file]\n" );
            return;
        }

        StringBuffer sBuffer = new StringBuffer(1024);
		liman.lim_get_verstr(sBuffer);
		System.out.println();
		System.out.println("Version: " + sBuffer.toString());

	    szProduct = args[0];
        szLicFile = args[1];
        new testapp().run();
    }

	/**
	/brief Handle specified error by displaying its description to stdout
	/param pEnv A LIMAN environment instance
	/param pnErr An integer reference to the error code
	*/
    public void onError(SWIGTYPE_p_LIMENV pEnv, int [] pnErr)

    {

        StringBuffer sbuf = new StringBuffer(255);
        if (pnErr[0] != liman.LIM_OK)
        {
            sbuf.setLength(0);
            liman.lim_get_errmsg(pEnv, pnErr[0], sbuf);
            System.out.println("Error " + pnErr[0] + ": " + sbuf.toString());
        }
    }

    /*
     * Entry point
     */
    @Override public void run()
    {
        SWIGTYPE_p_LIMENV pEnv;
        SWIGTYPE_p_LIMLIC pLic;
        int [] pnErr = new int [1];
        int [] keyLen = new int [1];
        pnErr[0] = liman.LIM_OK;

        System.out.printf("Application is using license file '%s'\n", szLicFile);

        /* get license */
        pEnv = liman.lim_create_env(2048, 256, pnErr);
        if (pnErr[0] != liman.LIM_OK) {

            onError(pEnv, pnErr);

        }

        if (0 > 1) {
			// load public key hardcoded in the app
			String szPubKey = "F4FFF999C2D782A1B73BDDF48187A70B6C2418C20B02A8BD862A490DE798D7C20556976550C47C45D9650E312308CED84CAC22691B4AD0C77775524D98E81D46578C9228F2CCB3C8F78AFBDA68D598031DA7FEBC46DF4D8CA43F3A4DAF0BC65AB14D87BC113291BB6F786D35A713EFF2B4014DFD6CAFFE9BBEBEA328CEC71BE15230AC41845B52A332CC1248B63C26F38B6CEE6C28904EA845A85A61F7EBA29E71C1499DBD2B272A5A5334BD29D97AB91490C8A3D2BC4B35738DF8AB1840DEB8106422968995735379AD46DB9EBD4E421114FD272075696831BB87E2822159AD0633A8F4299B9EAE92F381784F409C46E08C5C5856AA17D831825C25C70A98E3";
            pnErr[0] = liman.lim_load_public_key(pEnv, szPubKey);
            if (pnErr[0] != liman.LIM_OK) { onError(pEnv, pnErr);}
            System.out.println("Load public key.. ok.");
        }
        else {
			// alternatively,  read public key from file
			System.out.print("Reading public key..");
            pnErr[0] = liman.lim_read_public_key(pEnv, "../keys/" + szProduct + "/" + szProduct+ "_public.key");
            if (pnErr[0] != liman.LIM_OK) { onError(pEnv, pnErr);}
            System.out.println("ok.");
        }

        if (pnErr[0] != liman.LIM_OK) { onError(pEnv, pnErr);}

		// create a license
		System.out.print("Creating a license instance from license file " + szLicFile + "\n");
        pLic = liman.lim_create_lic_fromfile(pEnv, szLicFile, pnErr);
        if (pnErr[0] != liman.LIM_OK) {
            onError(pEnv, pnErr);
        }

        System.out.printf("Application License id: '%s'\n", liman.lim_lic_id(pLic));
        pnErr[0] = liman.lim_is_verified(pLic);

        if (pnErr[0] != liman.LIM_OK) {
            onError(pEnv, pnErr);
        }

        if (pnErr[0] != liman.LIM_OK) {
            System.out.println("Warning: License is not verified");
            onError(pEnv, pnErr);
        }
        else {
            System.out.println("Application License is verified!");
        }

        String product_id = new String(szProduct);

        if (product_id.equals(liman.lim_product_id(pLic)) == false) {
            pnErr[0] = liman.LIMERR_INVALID_PRODUCT_CODE;
            onError(pEnv, pnErr);
        }

        if (liman.lim_size_id(pLic) == liman.LIMSIZE_DEMO) {
            System.out.println("Application is using demo license.");
            onError(pEnv, pnErr);
        }

        System.out.printf("Application License version is %d.%d.\n", liman.lim_major_version(pLic), liman.lim_minor_version(pLic));

        // display license size and impose limitations accordingly

        if (liman.lim_platform_id(pLic) != liman.LIMPLAT_WIN32X86) {
            pnErr[0] = liman.LIMERR_INVALID_PLATFORM;
            onError(pEnv, pnErr);
        }
        else
        {
            System.out.printf("Application License is valid for platform-id %d.\n", liman.lim_platform_id(pLic));
        }

        // display license size and impose limitations accordingly
        System.out.printf("Application License type is %d.\n", liman.lim_type_id(pLic));
        if (liman.lim_type_id(pLic) == liman.LIMTYPE_EDUCATIONAL) {
			System.out.printf("License is good for educational institutions\n");
            // INSERT CODE
        }

        // display license size and impose limitations accordingly

        System.out.printf("Application License size is %d.\n", liman.lim_size_id(pLic));
        if (liman.lim_size_id(pLic) == liman.LIMSIZE_DEMO) {
			System.out.printf("License is good for demo purposes\n");
            // INSERT CODE
        }

        {

            //! [Availability of Sort Functions]

            // 0'th feature slot specifies which sort methods are avaiable under given license key
            if (0 < liman.lim_has_feature(pLic, slotIdSorting,  liman.LIMOPT_FEATURE_0)) {
                System.out.printf("Quick Sort Feature is available\n");
            }

            if (0 < liman.lim_has_feature(pLic, slotIdSorting,  liman.LIMOPT_FEATURE_1)) {
                System.out.printf("Merge Sort Feature is available\n");
            }

            if (0 < liman.lim_has_feature(pLic, slotIdSorting,  liman.LIMOPT_FEATURE_2)) {
                System.out.printf("Heap Sort Feature is available\n");
            }

            if (0 < liman.lim_has_feature(pLic, slotIdSorting,  liman.LIMOPT_FEATURE_3)) {
                System.out.printf("Insertion Sort Feature is available\n");
            }

            if (0 < liman.lim_has_feature(pLic, slotIdSorting,  liman.LIMOPT_FEATURE_4)) {
                System.out.printf("Intro Sort Feature is available\n");
            }

            if (0 < liman.lim_has_feature(pLic, slotIdSorting,  liman.LIMOPT_FEATURE_5)) {
                System.out.printf("Block Sort Feature is available\n");
            }

            if (0 < liman.lim_has_feature(pLic, slotIdSorting,  liman.LIMOPT_FEATURE_6)) {
                System.out.printf("Shell Sort Feature is available\n");
            }

            if (0 < liman.lim_has_feature(pLic, slotIdSorting,  liman.LIMOPT_FEATURE_7)) {
                System.out.printf("Bubble Sort Feature is available\n");
            }

            //! [Availability of Sort Functions]

        }

        // 1'st slot specifies which functions are avaiable under given license key
        {
            // run function 0
            pnErr[0] = app_function0(pLic);
            onError(pEnv, pnErr);

            // run function 1
            pnErr[0] = app_function1(pLic);
            onError(pEnv, pnErr);

            // run function 2
            pnErr[0] = app_function2(pLic);
            onError(pEnv, pnErr);
        }


        liman.lim_delete_lic(pLic);

        liman.lim_delete_env(pEnv);

        return;

    }

}
